<?php
/**
 * Daily Rebate Payout Cron Job
 * Runs daily at 12:01 AM to add previous day's rebate amounts to user wallets
 * 
 * Schedule: 1 0 * * * php /path/to/daily_rebate_payout_cron.php
 */

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone
date_default_timezone_set("Asia/Kolkata");

// Include database connection
require_once("../../conn.php");
require_once("rebate_functions.php");

// Log file for cron execution
$logFile = __DIR__ . "/cron_logs/daily_rebate_payout.log";
$lockFile = __DIR__ . "/cron_logs/daily_rebate_payout.lock";

// Create logs directory if it doesn't exist
if (!is_dir(__DIR__ . "/cron_logs")) {
    mkdir(__DIR__ . "/cron_logs", 0755, true);
}

// Function to write to log
function writeLog($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    echo $logMessage;
}

// Check if cron is already running
if (file_exists($lockFile)) {
    $lockTime = filemtime($lockFile);
    $currentTime = time();
    
    // If lock file is older than 30 minutes, remove it (assume previous cron failed)
    if (($currentTime - $lockTime) > 1800) {
        unlink($lockFile);
        writeLog("WARNING: Removed stale lock file");
    } else {
        writeLog("ERROR: Cron is already running. Exiting.");
        exit(1);
    }
}

// Create lock file
file_put_contents($lockFile, time());

writeLog("=== Daily Rebate Payout Cron Started ===");

try {
    // Get yesterday's date
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    writeLog("Processing rebates for date: $yesterday");
    
    // Get all users who have rebates for yesterday
    $usersQuery = "SELECT DISTINCT user_id, 
                          SUM(rebate_amount) as total_rebate,
                          COUNT(*) as bet_count
                   FROM bet_rebates 
                   WHERE DATE(created_at) = '$yesterday'
                   GROUP BY user_id
                   HAVING total_rebate > 0";
    
    $usersResult = $conn->query($usersQuery);
    
    if (!$usersResult) {
        throw new Exception("Failed to fetch users with rebates: " . $conn->error);
    }
    
    $totalUsers = $usersResult->num_rows;
    $totalRebateAmount = 0;
    $processedUsers = 0;
    $failedUsers = 0;
    
    writeLog("Found $totalUsers users with rebates for $yesterday");
    
    if ($totalUsers > 0) {
        // Start transaction
        $conn->begin_transaction();
        
        while ($userRow = $usersResult->fetch_assoc()) {
            $userId = $userRow['user_id'];
            $userRebateAmount = $userRow['total_rebate'];
            $betCount = $userRow['bet_count'];
            
            try {
                // Get current user balance
                $balanceQuery = "SELECT motta FROM shonu_kaichila WHERE balakedara = '$userId'";
                $balanceResult = $conn->query($balanceQuery);
                
                if ($balanceResult && $balanceResult->num_rows > 0) {
                    $balanceRow = $balanceResult->fetch_assoc();
                    $currentBalance = $balanceRow['motta'];
                    $newBalance = $currentBalance + $userRebateAmount;
                    
                    // Update user balance
                    $updateQuery = "UPDATE shonu_kaichila 
                                   SET motta = '$newBalance' 
                                   WHERE balakedara = '$userId'";
                    
                    if ($conn->query($updateQuery)) {
                        // Mark rebates as paid
                        $markPaidQuery = "UPDATE bet_rebates 
                                         SET is_paid = 1, paid_at = NOW() 
                                         WHERE user_id = '$userId' 
                                         AND DATE(created_at) = '$yesterday'
                                         AND (is_paid IS NULL OR is_paid = 0)";
                        
                        if ($conn->query($markPaidQuery)) {
                            $totalRebateAmount += $userRebateAmount;
                            $processedUsers++;
                            
                            writeLog("SUCCESS: User $userId - Added ₹$userRebateAmount rebate ($betCount bets) - New balance: ₹$newBalance");
                        } else {
                            writeLog("ERROR: Failed to mark rebates as paid for user $userId: " . $conn->error);
                            $failedUsers++;
                        }
                    } else {
                        writeLog("ERROR: Failed to update balance for user $userId: " . $conn->error);
                        $failedUsers++;
                    }
                } else {
                    writeLog("ERROR: User $userId not found in shonu_kaichila table");
                    $failedUsers++;
                }
                
            } catch (Exception $e) {
                writeLog("ERROR: Exception processing user $userId: " . $e->getMessage());
                $failedUsers++;
            }
        }
        
        // Commit transaction if all successful, rollback if any failed
        if ($failedUsers == 0) {
            $conn->commit();
            writeLog("SUCCESS: Transaction committed successfully");
        } else {
            $conn->rollback();
            writeLog("ERROR: Transaction rolled back due to $failedUsers failed users");
        }
        
    } else {
        writeLog("INFO: No users found with rebates for $yesterday");
    }
    
    // Add is_paid and paid_at columns if they don't exist
    $alterTableQuery = "ALTER TABLE bet_rebates 
                       ADD COLUMN is_paid TINYINT(1) DEFAULT 0,
                       ADD COLUMN paid_at TIMESTAMP NULL DEFAULT NULL";
    
    // This will fail if columns already exist, but that's fine
    @$conn->query($alterTableQuery);
    
    writeLog("=== Daily Rebate Payout Summary ===");
    writeLog("Date Processed: $yesterday");
    writeLog("Total Users Found: $totalUsers");
    writeLog("Users Processed Successfully: $processedUsers");
    writeLog("Users Failed: $failedUsers");
    writeLog("Total Rebate Amount Paid: ₹$totalRebateAmount");
    writeLog("=== Cron Completed Successfully ===");
    
} catch (Exception $e) {
    writeLog("FATAL ERROR: " . $e->getMessage());
    writeLog("Stack trace: " . $e->getTraceAsString());
    
    // Rollback transaction if active
    if ($conn->inTransaction ?? false) {
        $conn->rollback();
        writeLog("Transaction rolled back due to fatal error");
    }
    
    exit(1);
    
} finally {
    // Close database connection
    if ($conn) {
        $conn->close();
    }
    
    // Remove lock file
    if (file_exists($lockFile)) {
        unlink($lockFile);
    }
    
    writeLog("=== Daily Rebate Payout Cron Ended ===\n");
}

// Success
exit(0);
?>
